﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;

namespace xamMenuFun
{
    /// <summary>
    /// Class encapsulating manipulation of group panel data
    /// </summary>
    public class GroupPanelsBusinessLogic:INotifyPropertyChanged
    {
        #region Fields

        private ObservableCollection<GroupPanelData> _groupPanelList;

        #endregion Fields

        #region Properties

        /// <summary>
        /// Gets/sets the visible group panel observable collection.
        /// </summary>
        public ObservableCollection<GroupPanelData> VisibleGroupPanelList
        {
            get
            {
                ObservableCollection<GroupPanelData> visibleGroupPanelList = null;
                if (_groupPanelList != null)
                {
                    var visibleMenuItems = from data in _groupPanelList
                                           where data.IsVisible
                                           select data;
                    visibleGroupPanelList = new ObservableCollection<GroupPanelData>(visibleMenuItems);
                }
                return visibleGroupPanelList;                
            }
            set 
            { 
                _groupPanelList = value;
                OnPropertyChanged("VisibleGroupPanelList");                
            }            
        }

        /// <summary>
        /// Gets all group panels regarless of whether they are visible or not.
        /// </summary>
        public List<GroupPanelData> GroupPanelList
        {
            get
            {
                return _groupPanelList.ToList<GroupPanelData>();
            }
        }

        #endregion Properties

        #region Constructors

        public GroupPanelsBusinessLogic()
        {            
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Create and add a <see cref="GroupPanelData"/> to the observable collection.
        /// </summary>
        /// <param name="panelTitle"></param>
        /// <param name="locationIndex"></param>
        /// <param name="parentLocationIndex"></param>
        public void AddGroupPanel(string panelTitle, int locationIndex, int parentLocationIndex)
        {
            GroupPanelData panelData = new GroupPanelData(panelTitle, locationIndex, parentLocationIndex);
            AddGroupPanel(panelData);
        }

        /// <summary>
        ///  Add a <see cref="GroupPanelData"/> to the observable collection.
        /// </summary>
        /// <param name="panelData"></param>
        public void AddGroupPanel(GroupPanelData panelData)
        {
            if (_groupPanelList == null)
            {
                _groupPanelList = new ObservableCollection<GroupPanelData>();
            }

            _groupPanelList.Add(panelData);
        }

        /// <summary>
        /// Gets the group panel observable collection.
        /// </summary>
        /// <returns></returns>
        public ObservableCollection<GroupPanelData> GetGroupPanels()
        {             
            GroupPanelData parentData = new GroupPanelData("Item Data", 0, -1);
            parentData.AddChild(new GroupPanelData("Comments", 0, 0));
            parentData.AddChild(new GroupPanelData("Notes", 1, 0));
            parentData.AddChild(new GroupPanelData("Special Instructions", 2, 0));
            AddGroupPanel(parentData);
            AddGroupPanel("Inventory Data", 1, -1);
            AddGroupPanel("Location Data", 2, -1);
            parentData = new GroupPanelData("Misc Data", 3, -1);            
            parentData.AddChild(new GroupPanelData("Site", 0, 3));
            parentData.AddChild(new GroupPanelData("Pricing", 1,3));
            AddGroupPanel(parentData);
            parentData = new GroupPanelData("Image Data", 4, -1);
            parentData.AddChild(new GroupPanelData("Comments", 0, 4));
            AddGroupPanel(parentData);
            AddGroupPanel("Bill of Material Data", 5, -1);
            return _groupPanelList;
        }

        #endregion Methods

        #region INotifyPropertyChanged Members

        public event PropertyChangedEventHandler PropertyChanged;

        private void OnPropertyChanged(String info)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(info));
            }
        }

        #endregion INotifyPropertyChanged Members
    }

    /// <summary>
    /// Class encapsulating the group panel data.
    /// </summary>
    public class GroupPanelData : INotifyPropertyChanged
    {
        #region Constants

        /// <summary>
        /// For the <see cref="GroupPanelData"/>, an indicator that an element is a root element
        /// and not a child.
        /// </summary>
        public static readonly int RootElementIndicator = -1;

        #endregion Constants

        #region Fields

        private string _groupPanelTitle;
        private int _locationIndex;       
        private int _parentLocationIndex; // -1 => root element
        private ObservableCollection<GroupPanelData> _childGroupPanelList;
        private bool _hasChildren;
        private bool _isVisible;
        private bool _isHighlighted;

        #endregion Fields

        #region Properties

        /// <summary>
        /// Gets/sets the Title.
        /// </summary>
        public string Title
        {
            get { return _groupPanelTitle; }
            set
            {
                if (_groupPanelTitle != value)
                {
                    _groupPanelTitle = value;
                    OnPropertyChanged("Title");
                }
            }
        }

        /// <summary>
        /// The location index
        /// </summary>
        public int LocationIndex
        {
            get { return _locationIndex; }
            set { _locationIndex = value; }
        }

        /// <summary>
        /// When a child menu item, the parents location index
        /// </summary>
        public int ParentLocationIndex
        {
            get { return _parentLocationIndex; }
            set { _parentLocationIndex = value; }
        }

        /// <summary>
        /// Flag indicating if a group panel has child panels
        /// </summary>
        public bool HasChildren
        {
            get 
            {
                bool hasChildren = false;
                if (ChildGroupPanelList != null && ChildGroupPanelList.Count > 0)
                {
                    hasChildren = true;
                }
                return hasChildren; 
            }
            set 
            { 
                _hasChildren = value;
                OnPropertyChanged("HasChildren");
            }
        }

        /// <summary>
        /// Gets/sets whether a group panel is visible
        /// </summary>
        public bool IsVisible
        {
            get
            {
                return _isVisible;
            }
            set
            {
                _isVisible = value;
                OnPropertyChanged("IsVisible");
            }
        }

        /// <summary>
        /// Gets/sets whether a group panel is the currently highlighted item in the navigator.
        /// </summary>
        public bool IsHighlighted
        {
            get
            {
                return _isHighlighted;
            }
            set
            {
                _isHighlighted = value;
                OnPropertyChanged("IsHighlighted");
            }
        }

        /// <summary>
        /// Gets whether group panel has parent.
        /// </summary>
        public bool HasParent
        {
            get
            {
                bool hasParent = false;
                if (_parentLocationIndex != -1)
                {
                    hasParent = true;
                }
                return hasParent;
            }
        }

        /// <summary>
        /// Gets/sets the collection of child group panels
        /// </summary>
        public ObservableCollection<GroupPanelData> ChildGroupPanelList
        {
            get 
            {
                return _childGroupPanelList; 
            }
            set
            {
                _childGroupPanelList = value;
            }
        }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Instantiate a <see cref="GroupPanelData"/> instance.
        /// </summary>
        /// <param name="title"></param>
        /// <param name="locationIndex"></param>
        /// <param name="parentLocationIndex"></param>
        public GroupPanelData(string title, int locationIndex, int parentLocationIndex)
        {
            Title = title;
            _locationIndex = locationIndex;
            _parentLocationIndex = parentLocationIndex;
            _isVisible = true;
            _isHighlighted = false;
        }


        /// <summary>
        /// Instantiate a <see cref="GroupPanelData"/> instance. 
        /// </summary>
        /// <param name="title"></param>
        /// <param name="childPanels"></param>
        public GroupPanelData(string title, int locationIndex, int parentLocationIndex, ObservableCollection<GroupPanelData> childPanels):this(title, locationIndex, parentLocationIndex)
        {            
            ChildGroupPanelList = childPanels;
        }

        #endregion Constructors  
      
        #region Method

        /// <summary>
        /// Create and add the <see cref="GroupPanelData"/> as a child of this group panel data.
        /// </summary>
        /// <param name="childTitle"></param>
        /// <param name="locationIndex"></param>
        public void AddChild(string childTitle, int locationIndex)
        {
            GroupPanelData childData = new GroupPanelData(childTitle, locationIndex, -1);
            AddChild(childData);
        }

        /// <summary>
        /// Add the <see cref="GroupPanelData"/> as a child of this group panel data.
        /// </summary>
        /// <param name="panelData"></param>
        public void AddChild(GroupPanelData panelData)
        {
            if (_childGroupPanelList == null)
            {
                _childGroupPanelList = new ObservableCollection<GroupPanelData>();
            }
            _childGroupPanelList.Add(panelData);
        }

        #endregion Method

        #region INotifyPropertyChanged Members

        public event PropertyChangedEventHandler PropertyChanged;

        private void OnPropertyChanged(String info)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(info));
            }
        }
        #endregion INotifyPropertyChanged Members
    }

}
