﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

using Infragistics.Controls.Menus;
using Infragistics.Windows;

namespace xamMenuFun
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {        
        #region Fields

        private bool _buttonClicked = false;
        private XamMenuItem _lastClicked = null;
        private XamMenuItem _currentHighlighted = null;        
        private List<HierarchicalElementLocation> _locations = new List<HierarchicalElementLocation>();
        private GroupPanelsBusinessLogic _businessLogic;
        private double _bottomPanelHeight;
        private bool _ignoreOneScrollEvent = false;

        #endregion Fields

        #region Constructors

        public MainWindow()
        {
            _businessLogic = new GroupPanelsBusinessLogic();
            this.DataContext = _businessLogic;
            InitializeComponent();
            //_businessLogic.VisibleGroupPanelList = _businessLogic.GetGroupPanels();
        }

        #endregion Constructors

        #region Methods

        private void WireExpanderEvents(Grid grid, bool wire)
        {
            foreach (UIElement child in grid.Children)
            {
                Expander expander = child as Expander;
                if (expander != null)
                {
                    if (wire)
                    {
                        // Wire event to capture when expanders are expanded/collapsed
                        expander.SizeChanged += new SizeChangedEventHandler(expander_SizeChanged);
                    }
                    else
                    {
                        // Unwire event to capture when expanders are expanded/collapsed
                        expander.SizeChanged -= new SizeChangedEventHandler(expander_SizeChanged);
                    }

                    // ASSUMPTION: There can be two and only two levels of collapsable panels. 
                    //             Therefore only need to check this expander for child expanders
                    //             and no need for recursion.                    
                    object[] childExpanders = UIElementTreeHelper.FindControls(expander, typeof(Expander), 0);
                    if (childExpanders != null && childExpanders.Length > 0)
                    {
                        // Get the Grid the child expanders are part of
                        Grid childGrid = expander.Content as Grid;
                        if (childGrid != null)
                        {
                            foreach (Expander childExpander in childExpanders)
                            {
                                if (wire)
                                {
                                    // Wire event to capture when expanders are expanded/collapsed
                                    childExpander.SizeChanged += new SizeChangedEventHandler(expander_SizeChanged);
                                }
                                else
                                {
                                    // Unwire event to capture when expanders are expanded/collapsed
                                    childExpander.SizeChanged -= new SizeChangedEventHandler(expander_SizeChanged);
                                }
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Get the location of all expanders.
        /// </summary>
        /// <param name="grid"></param>
        private void ProcessLocationData(Grid grid)
        {
            double location = 0;
            List<HierarchicalElementLocation> childLocations = null;
            ObservableCollection<GroupPanelData> groupPanelList = new ObservableCollection<GroupPanelData>();
            GroupPanelData parentData = null;            

            for (int parentIndex = 0; parentIndex < grid.Children.Count; parentIndex++)            
            {
                UIElement parent = grid.Children[parentIndex];
                Expander groupPanel = parent as Expander;
                if (groupPanel != null)
                {                     
                    // Need to account for the margins (if it is visible)
                    location = location + ((groupPanel.Visibility == System.Windows.Visibility.Visible) ? groupPanel.Margin.Top : 0);

                    // Create the parent group panel data
                    parentData = new GroupPanelData(groupPanel.Header.ToString(), parentIndex, GroupPanelData.RootElementIndicator);
                    parentData.IsVisible = (groupPanel.Visibility == System.Windows.Visibility.Visible) ? true : false;

                    // ASSUMPTION: There can be two and only two levels of collapsable panels. 
                    //             Therefore only need to check this expander for child expanders
                    //             and no need for recursion.                    
                    object[] childGroupPanels = UIElementTreeHelper.FindControls(groupPanel, typeof(Expander), 0);
                    if (childGroupPanels != null && childGroupPanels.Length > 0)
                    {
                        childLocations = new List<HierarchicalElementLocation>();
                        GroupPanelData childData = null;
                        for (int childIndex = 0; childIndex < childGroupPanels.Length; childIndex++)
                        {
                            Expander childGroupPanel = childGroupPanels[childIndex] as Expander;

                            // Create the child group panel data
                            childData = new GroupPanelData(childGroupPanel.Header.ToString(), childIndex, parentIndex);
                            childData.IsVisible = (childGroupPanel.Visibility == System.Windows.Visibility.Visible) ? true : false;
                            parentData.AddChild(childData);

                            // The child element uses the same location as its parent. The offset will be used to position.
                            // Return the general transform for the child group panel
                            // relative to its parent (=> parent group panel)
                            GeneralTransform generalTransform = childGroupPanel.TransformToAncestor(groupPanel);

                            // Retrieve the point value relative to the parent.
                            Point currentPoint = generalTransform.Transform(new Point(0, 0));
                            double offset = currentPoint.Y;

                            HierarchicalElementLocation childLocation = new HierarchicalElementLocation(childGroupPanel.Name, 
                                                                                                        location, 
                                                                                                        offset,
                                                                                                        childData.IsVisible);
                            childLocations.Add(childLocation);
                        }

                    }

                    // Add to the location data                    
                    _locations.Add(childLocations == null ? new HierarchicalElementLocation(groupPanel.Name, location, location, parentData.IsVisible) : new HierarchicalElementLocation(groupPanel.Name, location, location, parentData.IsVisible, childLocations));
                    location += groupPanel.ActualHeight + ((groupPanel.Visibility == System.Windows.Visibility.Visible) ? groupPanel.Margin.Bottom : 0);
                    
                    // parentIndex is 0 based, count is 1 based so -2
                    if (parentIndex == (grid.Children.Count - 2))
                    {
                        _bottomPanelHeight = groupPanel.ActualHeight - (2 * (groupPanel.Margin.Top + groupPanel.Margin.Bottom));
                    }

                    // Add to the groupPanelList
                    groupPanelList.Add(parentData);

                    // Child locations might have been set by a prior Expander, reset to null
                    childLocations = null;
                }
            }

            // Set the view model for the menu to the groupPanelList
            this._businessLogic.VisibleGroupPanelList = groupPanelList;
        }

        private void ResetPanelVisibility()
        {
            Grid svGrid = _scrollViewer.Content as Grid;
            if (svGrid != null)
            {
                foreach (GroupPanelData data in this._businessLogic.GroupPanelList)
                {
                    UIElement child = svGrid.Children[data.LocationIndex] as UIElement;
                    Expander expander = child as Expander;
                    if (expander != null)
                    {
                        if (data.IsVisible)
                        {
                            expander.Visibility = System.Windows.Visibility.Visible;
                        }
                        else
                        {
                            expander.Visibility = System.Windows.Visibility.Collapsed;
                        }
                    }
                }
            }
        }

        private void ResetAllPanelLocations()
        {
            Grid svGrid = _scrollViewer.Content as Grid;
            if (svGrid != null)
            {
                _locations.Clear();
                ProcessLocationData(svGrid);
            }
        }

        private int GetLocationIndex()
        {
            int index = 0;
            int numInvisiblePanels = 0;
            for (int i = 0; i < _locations.Count; i++)
            {
                if (_locations[i].IsGroupPanelVisible)
                {
                    if (_scrollViewer.VerticalOffset >= _locations[i].Location)
                    {
                        index = i - numInvisiblePanels;
                    }
                    else
                    {
                        break;
                    }
                }
                else
                {
                    numInvisiblePanels++;
                }
            }

            return index;
        }

        private void MakeFirstMenuItemSelected()
        {
            IList<FrameworkElement> menuItems = UIElementTreeHelper.GetDescendantsFromType(_navigatorMenu, typeof(XamMenuItem));
            XamMenuItem xmi = menuItems[0] as XamMenuItem;
            DecorateMenu(xmi);
            _lastClicked = xmi;            
        }

        #endregion Methods

        #region Event handlers

        void expander_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            // Need to reset all panel locations
            ResetAllPanelLocations();
        }

        /// <summary>
        /// Not currently wired to anything
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void xamNavigator_ItemClicked(object sender, Infragistics.Controls.Menus.ItemClickedEventArgs e)
        {
            //XamMenuItem mi = e.Item;
            //mi.IsSubmenuOpen = false;
            GroupPanelData item = e.Item.Header as GroupPanelData;
            
            MessageBox.Show("Item clicked: " + item.Title);
        }

        private void NavigatorItemLabel_Loaded(object sender, RoutedEventArgs e)
        {
            XamMenuItem xmi = Utilities.GetAncestorFromType(sender as DependencyObject, typeof(XamMenuItem), true) as XamMenuItem;

            // Set tooltip
            TextBlock textBlock = sender as TextBlock;
            ToolTipService.SetInitialShowDelay(textBlock, 1000);            
            textBlock.ToolTip = textBlock.Text;

            // Wire events -- click means navigate to group panel, SubmenuOpened is to keep drop down from happening
            xmi.Click += new EventHandler(xmi_Click);
            xmi.SubmenuOpened +=new EventHandler(xmi_SubmenuOpened);
            xmi.SubmenuClosed += new EventHandler(xmi_SubmenuClosed);
            

        }

        void xmi_SubmenuClosed(object sender, EventArgs e)
        {
            _buttonClicked = false;
        }

        private void DecorateMenu(XamMenuItem mi)
        {
            // Decorate the correct menut item, if it is a child,
            // decorate its parent.
            XamMenuItem newMenuItemToHighlight = mi;            
            XamMenuItem parentMenuItem = mi.ParentXamMenuItem;
            if (parentMenuItem != null)
            {
                newMenuItemToHighlight = parentMenuItem;
            }

            GroupPanelData gpd;
            if (_currentHighlighted != null)
            {
                // Clear the current highlight
                gpd = _currentHighlighted.Header as GroupPanelData;
                gpd.IsHighlighted = false;
            }

            // Hightlight the new menu item
            gpd = newMenuItemToHighlight.Header as GroupPanelData;
            gpd.IsHighlighted = true;

            _currentHighlighted = newMenuItemToHighlight;
       
        }

         /// <summary>
        /// If the panel at the locationIndex is enabled and collapsed, expand it.
        /// </summary>
        /// <param name="uiElementIndex"></param>        
        private void ExpandGroupPanel(int uiElementIndex)
        {
            ExpandGroupPanel(uiElementIndex, -1);
        }

        /// <summary>
        /// If the panel at the locationIndex is enabled and collapsed, expand it. 
        /// Expand the child panel as indicated by the passed index.
        /// </summary>
        /// <param name="uiElementIndex"></param>
        /// <param name="childIndex">If not -1, process child</param>
        private void ExpandGroupPanel(int uiElementIndex, int childIndex)
        {            
            Grid svGrid = _scrollViewer.Content as Grid;
            if (svGrid != null)
            {
                UIElementCollection elements = svGrid.Children;
                UIElement uiElement = elements[uiElementIndex];
                Expander groupPanel = uiElement as Expander;

                // If the group panel is enabled and collapsed, expand it
                if (groupPanel.IsEnabled)
                {
                    if (!groupPanel.IsExpanded)
                    {
                        groupPanel.IsExpanded = true;
                        ResetAllPanelLocations();
                    }

                    if (childIndex != -1)
                    {
                        // ASSUMPTION: There can be two and only two levels of collapsable panels. 
                        //             Therefore only need to check this expander for child expanders
                        //             and no need for recursion.  

                        // Process child index
                        object[] childExpanders = UIElementTreeHelper.FindControls(groupPanel, typeof(Expander), 0);

                        if (childExpanders != null && childExpanders.Length > 0)
                        {
                            Expander childGroupPanel = childExpanders[childIndex] as Expander;
                            if (childGroupPanel.IsEnabled && !childGroupPanel.IsExpanded)
                            {
                                childGroupPanel.IsExpanded = true;
                                ResetAllPanelLocations();
                            }
                        }
                    }

                }
            }
        }       

        void xmi_Click(object sender, EventArgs e)
        {
            XamMenuItem mi = sender as XamMenuItem;

            // Only if we've clicked something new
            if (_lastClicked == null || (_lastClicked != null && _lastClicked != mi))
            {
                // Need to decorate the clicked menu item to indicate it is active                               
                DecorateMenu(mi);

                // Reset last clicked
                _lastClicked = mi;
                
                // Scroll to correct spot
                GroupPanelData item = mi.Header as GroupPanelData;                

                HierarchicalElementLocation parentLocation = null;
                HierarchicalElementLocation childOffset = null;               

                // Scroll 
                _ignoreOneScrollEvent = true;
                if (item.HasParent)
                {
                    // If the parent is collapsed, expand it (so we can see the child) 
                    ExpandGroupPanel(item.ParentLocationIndex, item.LocationIndex); 
                
                    parentLocation = _locations[item.ParentLocationIndex];
                    childOffset = parentLocation.Children[item.LocationIndex];
                    _scrollViewer.ScrollToVerticalOffset(parentLocation.Location + childOffset.Offset);
                }
                else
                {
                    // If the panel is collapsed, expand it
                    ExpandGroupPanel(item.LocationIndex);

                    // Don't be confused, this is a parent but use its LocationIndex.
                    parentLocation = _locations[item.LocationIndex];
                    _scrollViewer.ScrollToVerticalOffset(parentLocation.Location);
                }                
            }
        }

        private void ScrollToFirstItem()
        {
            HierarchicalElementLocation location = _locations[0];
            _scrollViewer.ScrollToVerticalOffset(location.Location);
        }

        void xmi_SubmenuOpened(object sender, EventArgs e)
        {            
            XamMenuItem mi = sender as XamMenuItem;
            if (mi.HasChildren && !_buttonClicked)
            {                
                mi.IsSubmenuOpen = false;
                xmi_Click(sender, e);
            }

            // Reset the button flag if it was true this go around
            if (_buttonClicked)
            {
                _buttonClicked = false;
            }
        }

        private void Button_Click(object sender, RoutedEventArgs e)
        {
            // Need a way to keep the xmi_SubmenuOpened event from processing
            // This is klugey way for now.
            _buttonClicked = true;

            // Drop down submenu
            XamMenuItem xmi = Utilities.GetAncestorFromType(sender as DependencyObject, typeof(XamMenuItem), true) as XamMenuItem;
            
            if (xmi.HasChildren)
            {
                xmi.IsSubmenuOpen = true;
            }
        }

        private void NavigatorSubitem_Loaded(object sender, RoutedEventArgs e)
        {
            XamMenuItem xmi = Utilities.GetAncestorFromType(sender as DependencyObject, typeof(XamMenuItem), true) as XamMenuItem;

            // Set tooltip
            TextBlock textBlock = sender as TextBlock;
            ToolTipService.SetInitialShowDelay(textBlock, 1000);
            textBlock.ToolTip = textBlock.Text;

            // Wire events -- click means navigate to group panel, SubmenuOpened is to keep drop down from happening
            xmi.Click += new EventHandler(xmi_Click);
        }

        private void _scrollViewer_Loaded(object sender, RoutedEventArgs e)
        {
            if (this.IsLoaded)
            {
                ScrollViewer scrollViewer = sender as ScrollViewer;
                Grid svGrid = scrollViewer.Content as Grid;

                // Going to want to use the RowDefinition for the location
                // as the Expander is padded at top. Need a complex object
                // since expanders can be nested.
                if (svGrid != null)
                {
                    // Process all Expander locations, after method finishes
                    // execution, the _location variable will have all
                    // the Expander locations.
                    ProcessLocationData(svGrid);

                    // Wire expander events
                    WireExpanderEvents(svGrid, true);

                    // Make first menu item selected
                    // NOTE: This does not scroll to the offset
                    MakeFirstMenuItemSelected();
                }
            }
        }

        private void Window_Loaded(object sender, RoutedEventArgs e)
        {
            //ProcessLocationData(_scrollViewer.Content as Grid);

            // Wire expander events
            //WireExpanderEvents(_scrollViewer.Content as Grid, true);

            // Make first menu item selected
            //MakeFirstMenuItemSelected();            

            // Need to wire here after the scroll viewer has been loaded
            _scrollViewer.SizeChanged += new SizeChangedEventHandler(_scrollViewer_SizeChanged);
            _scrollViewer.ScrollChanged += new ScrollChangedEventHandler(_scrollViewer_ScrollChanged); 
 
            
            
        }

        private void _scrollViewer_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            if (_emptyPanel != null)
                _emptyPanel.Height = _scrollViewer.ViewportHeight;
        }

        private void _scrollViewer_ScrollChanged(object sender, ScrollChangedEventArgs e)
        {
            if (!_ignoreOneScrollEvent)
            {
                // Determine what should be highlighted based on where the scroll is
                int locationIndex = GetLocationIndex();

                // Get the top level menu items
                XamMenu menu = _navigatorMenu;
                IList<FrameworkElement> menuItems = UIElementTreeHelper.GetDescendantsFromType(menu, typeof(XamMenuItem));

                // Get the menu item that should be highlighted
                XamMenuItem mi = menuItems[locationIndex] as XamMenuItem;

                DecorateMenu(mi);

                // If the last clicked was a sub-menu item under the menu item 
                // referenced by mi, don't reset the LastClicked
                if (_lastClicked != null)
                {
                    GroupPanelData childData = _lastClicked.Header as GroupPanelData;
                    GroupPanelData parentData = mi.Header as GroupPanelData;
                    if (childData.ParentLocationIndex != parentData.LocationIndex)
                    {
                        _lastClicked = mi;
                    }
                }
            }
            else
            {
                _ignoreOneScrollEvent = false;
            }
        }

        private void XamContextMenu_Opening(object sender, OpeningEventArgs e)
        {
            XamContextMenu contextMenu = sender as XamContextMenu;               
            contextMenu.Items.Clear();            

            // For each menu item add a context menu item
            Style menuItemStyle = this.Resources["PurpleMenuItemStyle"] as Style;
            XamMenuItem contextMenuItem;
            foreach (GroupPanelData data in this._businessLogic.GroupPanelList)
            {                
                contextMenuItem = new XamMenuItem();
                contextMenuItem.Style = menuItemStyle;
                contextMenuItem.Header = data.Title;
                contextMenuItem.IsCheckable = true;                
                contextMenuItem.IsChecked = data.IsVisible;                  
                contextMenuItem.Tag = data;

                // ASSUMPTION: There can be two and only two levels of collapsable panels. 
                //             Therefore only need to check this menu item for children
                //             and no need for recursion. 

                // Check for child panels, if found build the child menu items
                if (data.HasChildren)
                {
                    XamMenuItem childContextMenuItem;
                    foreach (GroupPanelData childData in data.ChildGroupPanelList)
                    {
                        childContextMenuItem = new XamMenuItem();
                        childContextMenuItem.Style = menuItemStyle;
                        childContextMenuItem.Header = childData.Title;
                        childContextMenuItem.IsCheckable = true;
                        childContextMenuItem.StaysOpenOnClick = true;
                        childContextMenuItem.IsChecked = childData.IsVisible;
                        childContextMenuItem.Tag = childData;
                        contextMenuItem.Items.Add(childContextMenuItem);
                    }
                }

                contextMenu.Items.Add(contextMenuItem);                
            }

            // Add a seperator
            XamMenuSeparator separator = new XamMenuSeparator();            
            contextMenu.Items.Add(separator);            

            // Add the 'Save for User' menu item
            contextMenuItem = new XamMenuItem();
            contextMenuItem.Style = menuItemStyle;
            contextMenuItem.Header = "Save for Session";            
            contextMenuItem.StaysOpenOnClick = false;
            contextMenuItem.Click += new EventHandler(contextMenuItem_Click);
            contextMenu.Items.Add(contextMenuItem);
        }

        void contextMenuItem_Click(object sender, EventArgs e)
        {
            XamMenuItem item = sender as XamMenuItem;
            bool hasChanges = false;
            bool noMenuItemsSelected = false;
            bool resetToFirstVisible = false;

            // Has to have at least one visible item            
            var visibleItems = from XamMenuItem mi in _navigatorContextMenu.Items.OfType<XamMenuItem>()
                               where mi.IsChecked
                               select mi;
            
            noMenuItemsSelected = visibleItems.Count<XamMenuItem>() < 1;

            if (!noMenuItemsSelected)
            {
                for (int i = 0; i < _navigatorContextMenu.Items.Count; i++)
                {
                    XamMenuItem menuItem = _navigatorContextMenu.Items[i] as XamMenuItem;

                    // If it is not one of the 'Save' menu items
                    if (menuItem != null &&
                        menuItem != item)
                    {
                        GroupPanelData data = menuItem.Tag as GroupPanelData;

                        // Did the visible state change
                        if (data.IsVisible != menuItem.IsChecked)
                        {
                            // Need to keep track if at least one menu item changed
                            hasChanges = true;

                            data.IsVisible = menuItem.IsChecked;

                            // If the currently selected menu item is being hidden, move
                            // the selected to the first visible element
                            if (!data.IsVisible &&
                               (this._currentHighlighted.Header == data))
                            {                                
                                resetToFirstVisible = true;
                            }
                        }
                    }
                }

                if (hasChanges)
                {                    
                    // Show and hide panels according to settings
                    ResetPanelVisibility();

                    // Clear the current highlighted and last clicked (they will be made undefined by reset of VisibleGroupPanelList)
                    this._currentHighlighted = null;
                    this._lastClicked = null;

                    // Reset the VisibleGroupPaneList to make it update                    
                    ResetAllPanelLocations();

                    if (resetToFirstVisible)
                    {
                        ScrollToFirstItem();
                    }
                    else
                    {
                        //ResetLastClickedHighlighted(gpdCurrentHighlighted, gpdLastClicked);
                        //DecorateMenu(_currentHighlighted);
                        //_ignoreOneScrollEvent = true;
                        //MakeFirstMenuItemSelected();
                        //ScrollToFirstItem();
                    }                   
                }
            }
            else
            {
                // Notify user that one item must be selected                
                MessageBox.Show("At least one group must be visible.");
            }
        }        

        /// <summary>
        /// Reset the _currentHighlighted and _lastClicked fields.
        /// </summary>
        /// <param name="gpdCurrentHighlighted"></param>
        /// <param name="gpdLastClicked"></param>
        private void ResetLastClickedHighlighted(GroupPanelData gpdCurrentHighlighted, GroupPanelData gpdLastClicked)
        {
            if (gpdCurrentHighlighted != null && gpdLastClicked != null)
            {
                IList<FrameworkElement> menuItems = UIElementTreeHelper.GetDescendantsFromType(_navigatorMenu, typeof(XamMenuItem));                
                foreach (XamMenuItem xmi in menuItems)
                {
                    if (xmi.Header == gpdCurrentHighlighted)
                    {
                        _currentHighlighted = xmi;
                    }

                    if (xmi.Header == gpdLastClicked)
                    {
                        _lastClicked = xmi;
                    }
                }                                
            }
        }

        #endregion Event handlers

        private void _navigatorMenu_Loaded(object sender, RoutedEventArgs e)
        {

        }
    }
}
